/*
 * Decompiled with CFR 0.152.
 */
package ch.serverrocket.database;

import ch.serverrocket.config.configfiles.ServerRocketConfig;
import com.hypixel.hytale.logger.HytaleLogger;
import java.io.File;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;

public final class DatabaseManager {
    private static final HytaleLogger LOGGER = HytaleLogger.forEnclosingClass();
    private static Connection connection;

    private DatabaseManager() {
    }

    public static void init(File dataDirectory, ServerRocketConfig.DatabaseConfig dbConfig) {
        try {
            ((HytaleLogger.Api)LOGGER.atInfo()).log("Initializing DatabaseManager...");
            String type = dbConfig.getType().toLowerCase();
            if (type.equals("sqlite")) {
                Class.forName("org.sqlite.JDBC");
                File dbDir = new File(dataDirectory, "data");
                if (!dbDir.exists()) {
                    dbDir.mkdirs();
                }
                File dbFile = new File(dbDir, "serverrocket.sqlite");
                ((HytaleLogger.Api)LOGGER.atInfo()).log("Opening SQLite DB at: " + dbFile.getAbsolutePath());
                connection = DriverManager.getConnection("jdbc:sqlite:" + dbFile.getAbsolutePath());
                ((HytaleLogger.Api)LOGGER.atInfo()).log("SQLite connection established successfully.");
            } else {
                if (!type.equals("mysql") && !type.equals("mariadb")) {
                    throw new IllegalArgumentException("Unsupported database type: " + type);
                }
                if (type.equals("mysql")) {
                    Class.forName("com.mysql.cj.jdbc.Driver");
                } else if (type.equals("mariadb")) {
                    Class.forName("org.mariadb.jdbc.Driver");
                }
                String url = String.format("jdbc:%s://%s:%d/%s?useSSL=false&allowPublicKeyRetrieval=true", type, dbConfig.getHost(), dbConfig.getPort(), dbConfig.getName());
                ((HytaleLogger.Api)LOGGER.atInfo()).log("Connecting to " + type + " at: " + dbConfig.getHost() + ":" + dbConfig.getPort());
                connection = DriverManager.getConnection(url, dbConfig.getUsername(), dbConfig.getPassword());
                ((HytaleLogger.Api)LOGGER.atInfo()).log(type + " connection established successfully.");
            }
            DatabaseManager.createTables(dbConfig.getType().toLowerCase());
            ((HytaleLogger.Api)LOGGER.atInfo()).log("Database schema verification completed.");
        }
        catch (Throwable t2) {
            ((HytaleLogger.Api)((HytaleLogger.Api)LOGGER.atSevere()).withCause(t2)).log("FATAL: Database initialization failed");
            throw new IllegalStateException("Database initialization failed", t2);
        }
    }

    private static void createTables(String type) throws Exception {
        boolean isSqlite = type.equals("sqlite");
        String autoIncrement = isSqlite ? "AUTOINCREMENT" : "AUTO_INCREMENT";
        try (Statement stmt = connection.createStatement();){
            stmt.execute("    CREATE TABLE IF NOT EXISTS Players (\n        players_uuid VARCHAR(36) PRIMARY KEY,\n        players_displayname VARCHAR(50),\n        players_firstjoin BIGINT,\n        players_lastseen BIGINT,\n        players_playtime BIGINT\n    );\n");
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS Duties (\n    duties_id INTEGER PRIMARY KEY %s,\n    duties_player VARCHAR(36) NOT NULL UNIQUE,\n    duties_status INTEGER NOT NULL DEFAULT 0,\n    duties_auto INTEGER NOT NULL DEFAULT 0,\n\n    FOREIGN KEY (duties_player)\n        REFERENCES Players(players_uuid)\n        ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS ReportsReason (\n    reportsreason_id INTEGER PRIMARY KEY %s,\n    reportsreason_name VARCHAR(100) NOT NULL UNIQUE,\n    reportsreason_description TEXT,\n    reportsreason_priority INTEGER NOT NULL DEFAULT 0,\n    reportsreason_status INTEGER NOT NULL DEFAULT 1\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS Reports (\n    reports_id INTEGER PRIMARY KEY %s,\n    reports_reporter VARCHAR(36) NOT NULL,\n    reports_reported VARCHAR(36) NOT NULL,\n    reports_staff VARCHAR(36),\n    reports_reason_id INTEGER NOT NULL,\n    reports_timestamp BIGINT NOT NULL,\n    reports_comment TEXT,\n    reports_status INTEGER NOT NULL DEFAULT 0,\n    reports_source_server VARCHAR(100) NOT NULL DEFAULT '',\n\n    FOREIGN KEY (reports_reporter) REFERENCES Players(players_uuid) ON DELETE CASCADE,\n    FOREIGN KEY (reports_reported) REFERENCES Players(players_uuid) ON DELETE CASCADE,\n    FOREIGN KEY (reports_staff) REFERENCES Players(players_uuid) ON DELETE SET NULL,\n    FOREIGN KEY (reports_reason_id) REFERENCES ReportsReason(reportsreason_id) ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS ModerationReasons (\n    modreason_id INTEGER PRIMARY KEY %s,\n    modreason_type VARCHAR(20) NOT NULL,\n    modreason_name VARCHAR(100) NOT NULL UNIQUE,\n    modreason_default_duration INTEGER,\n    modreason_status INTEGER NOT NULL DEFAULT 1\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS ModerationHistory (\n    mod_id INTEGER PRIMARY KEY %s,\n    mod_target VARCHAR(36) NOT NULL,\n    mod_staff VARCHAR(36) NOT NULL,\n    mod_type VARCHAR(20) NOT NULL,\n    mod_reason_id INTEGER,\n    mod_custom_reason TEXT,\n    mod_timestamp BIGINT NOT NULL,\n    mod_expiry BIGINT,\n    mod_active INTEGER NOT NULL DEFAULT 1,\n\n    FOREIGN KEY (mod_target) REFERENCES Players(players_uuid) ON DELETE CASCADE,\n    FOREIGN KEY (mod_staff) REFERENCES Players(players_uuid) ON DELETE CASCADE,\n    FOREIGN KEY (mod_reason_id) REFERENCES ModerationReasons(modreason_id) ON DELETE SET NULL\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS ReportAuditLog (\n    audit_id INTEGER PRIMARY KEY %s,\n    report_id INTEGER NOT NULL,\n    staff_uuid VARCHAR(36) NOT NULL,\n    action VARCHAR(100) NOT NULL,\n    details TEXT,\n    timestamp BIGINT NOT NULL,\n\n    FOREIGN KEY (report_id) REFERENCES Reports(reports_id) ON DELETE CASCADE,\n    FOREIGN KEY (staff_uuid) REFERENCES Players(players_uuid) ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute("CREATE TABLE IF NOT EXISTS Friends (\n    friends_uuid1 VARCHAR(36) NOT NULL,\n    friends_uuid2 VARCHAR(36) NOT NULL,\n    friends_status INTEGER NOT NULL DEFAULT 0,\n    friends_timestamp BIGINT NOT NULL,\n\n    PRIMARY KEY (friends_uuid1, friends_uuid2),\n    FOREIGN KEY (friends_uuid1) REFERENCES Players(players_uuid) ON DELETE CASCADE,\n    FOREIGN KEY (friends_uuid2) REFERENCES Players(players_uuid) ON DELETE CASCADE\n);\n");
            stmt.execute("CREATE TABLE IF NOT EXISTS FriendsBlocked (\n    blocker_uuid VARCHAR(36) NOT NULL,\n    blocked_uuid VARCHAR(36) NOT NULL,\n    blocked_timestamp BIGINT NOT NULL,\n\n    PRIMARY KEY (blocker_uuid, blocked_uuid),\n    FOREIGN KEY (blocker_uuid) REFERENCES Players(players_uuid) ON DELETE CASCADE,\n    FOREIGN KEY (blocked_uuid) REFERENCES Players(players_uuid) ON DELETE CASCADE\n);\n");
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS Stocks (\n    stock_id INTEGER PRIMARY KEY %s,\n    stock_name VARCHAR(50) NOT NULL UNIQUE,\n    stock_symbol VARCHAR(10) NOT NULL UNIQUE,\n    stock_owner VARCHAR(36),\n    stock_price DOUBLE NOT NULL DEFAULT 100.0,\n    stock_last_price DOUBLE NOT NULL DEFAULT 100.0,\n    stock_total_shares INTEGER NOT NULL DEFAULT 1000,\n    stock_available_shares INTEGER NOT NULL DEFAULT 1000,\n\n    FOREIGN KEY (stock_owner) REFERENCES Players(players_uuid) ON DELETE SET NULL\n);\n", autoIncrement));
            stmt.execute("CREATE TABLE IF NOT EXISTS PlayerStocks (\n    player_uuid VARCHAR(36) NOT NULL,\n    stock_id INTEGER NOT NULL,\n    shares INTEGER NOT NULL DEFAULT 0,\n    average_buy_price DOUBLE NOT NULL DEFAULT 0.0,\n    realized_profit DOUBLE NOT NULL DEFAULT 0.0,\n\n    PRIMARY KEY (player_uuid, stock_id),\n    FOREIGN KEY (stock_id) REFERENCES Stocks(stock_id) ON DELETE CASCADE\n);\n");
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS LimitOrders (\n    order_id INTEGER PRIMARY KEY %s,\n    stock_id INTEGER NOT NULL,\n    player_uuid VARCHAR(36) NOT NULL,\n    order_type VARCHAR(10) NOT NULL, -- BUY or SELL\n    amount INTEGER NOT NULL,\n    target_price DOUBLE NOT NULL,\n\n    FOREIGN KEY (stock_id) REFERENCES Stocks(stock_id) ON DELETE CASCADE,\n    FOREIGN KEY (player_uuid) REFERENCES Players(players_uuid) ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS StockSplitLog (\n    log_id INTEGER PRIMARY KEY %s,\n    stock_id INTEGER NOT NULL,\n    ratio_numerator INTEGER NOT NULL,\n    ratio_denominator INTEGER NOT NULL,\n    old_price DOUBLE NOT NULL,\n    new_price DOUBLE NOT NULL,\n    timestamp BIGINT NOT NULL,\n\n    FOREIGN KEY (stock_id) REFERENCES Stocks(stock_id) ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS FederalBankLog (\n    log_id INTEGER PRIMARY KEY %s,\n    stock_id INTEGER NOT NULL,\n    action_type VARCHAR(10) NOT NULL, -- BUY or SELL\n    amount INTEGER NOT NULL,\n    price DOUBLE NOT NULL,\n    timestamp BIGINT NOT NULL,\n\n    FOREIGN KEY (stock_id) REFERENCES Stocks(stock_id) ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS DividendLog (\n    log_id INTEGER PRIMARY KEY %s,\n    stock_id INTEGER NOT NULL,\n    total_amount DOUBLE NOT NULL,\n    amount_per_share DOUBLE NOT NULL,\n    timestamp BIGINT NOT NULL,\n\n    FOREIGN KEY (stock_id) REFERENCES Stocks(stock_id) ON DELETE CASCADE\n);\n", autoIncrement));
            stmt.execute("CREATE TABLE IF NOT EXISTS Economy (\n    economy_player VARCHAR(36) PRIMARY KEY,\n    economy_balance DOUBLE NOT NULL DEFAULT 0,\n    FOREIGN KEY (economy_player) REFERENCES Players(players_uuid) ON DELETE CASCADE\n);\n");
            try {
                DatabaseManager.ensurePlayerStocksColumns();
            }
            catch (Exception e) {
                ((HytaleLogger.Api)LOGGER.atWarning()).log("Could not migrate PlayerStocks columns", (Object)e);
            }
            stmt.execute(String.format("CREATE TABLE IF NOT EXISTS Transactions (\n    trans_id INTEGER PRIMARY KEY %s,\n    trans_sender VARCHAR(36),\n    trans_receiver VARCHAR(36),\n    trans_amount DOUBLE NOT NULL,\n    trans_timestamp BIGINT NOT NULL,\n    trans_type VARCHAR(20) NOT NULL,\n    trans_description TEXT,\n    trans_rolledback INTEGER NOT NULL DEFAULT 0,\n\n    FOREIGN KEY (trans_sender) REFERENCES Players(players_uuid) ON DELETE SET NULL,\n    FOREIGN KEY (trans_receiver) REFERENCES Players(players_uuid) ON DELETE SET NULL\n);\n", autoIncrement));
            try {
                DatabaseManager.ensureReportsSourceServerColumn(type);
            }
            catch (Exception e) {
                ((HytaleLogger.Api)LOGGER.atWarning()).log("Could not verify/migrate reports_source_server column", (Object)e);
            }
        }
    }

    private static void ensureReportsSourceServerColumn(String type) throws Exception {
        boolean isSqlite = type.equals("sqlite");
        if (isSqlite) {
            boolean exists = false;
            try (Statement stmt = connection.createStatement();
                 ResultSet rs = stmt.executeQuery("PRAGMA table_info(Reports);");){
                while (rs.next()) {
                    String col = rs.getString("name");
                    if (!"reports_source_server".equalsIgnoreCase(col)) continue;
                    exists = true;
                    break;
                }
            }
            if (!exists) {
                stmt = connection.createStatement();
                try {
                    stmt.executeUpdate("ALTER TABLE Reports ADD COLUMN reports_source_server VARCHAR(100) NOT NULL DEFAULT '';");
                }
                finally {
                    if (stmt != null) {
                        stmt.close();
                    }
                }
            }
        } else {
            try (Statement stmt = connection.createStatement();){
                try {
                    stmt.executeUpdate("ALTER TABLE Reports ADD COLUMN reports_source_server VARCHAR(100) NOT NULL DEFAULT '';");
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
    }

    private static void ensurePlayerStocksColumns() throws Exception {
        String[] columns;
        for (String col : columns = new String[]{"average_buy_price", "realized_profit"}) {
            try (Statement stmt = connection.createStatement();){
                try {
                    stmt.executeUpdate("ALTER TABLE PlayerStocks ADD COLUMN " + col + " DOUBLE NOT NULL DEFAULT 0.0;");
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
        }
    }

    public static Connection getConnection() {
        if (connection == null) {
            throw new IllegalStateException("DatabaseManager not initialized. Call DatabaseManager.init() in setup().");
        }
        return connection;
    }

    public static void shutdown() {
        try {
            if (connection != null && !connection.isClosed()) {
                connection.close();
                ((HytaleLogger.Api)LOGGER.atInfo()).log("Database connection closed safely.");
            }
        }
        catch (Exception e) {
            ((HytaleLogger.Api)LOGGER.atWarning()).log("Error while closing database connection", (Object)e);
        }
    }
}

